/**
 * Project: Minerva KB
 * Copyright: 2015-2016 @KonstruktStudio
 */
(function($) {
    'use strict';

    var GLOBAL_DATA = window.MinervaKB;
    var ui = window.MinervaUI;

    if (!GLOBAL_DATA || !ui) {
        return;
    }

    var i18n = GLOBAL_DATA.i18n;
    var settings = GLOBAL_DATA.settings;

    // global editor reference
    var editor = null;

    // global editor state
    var store = {
        emailTagsCache: {
            base: null
        }
    };

    // create popup instance
    var popup = new ui.Popup();

    popup.bindEvents({
        'click .fn-mkb-popup-close': popup.close.bind(popup),
        'click .js-mkb-insert-email-tag': handleEmailTagInsert
    });

    /**
     * Renders all he available email tags
     */
    function renderEmailTagsList(emailTemplate) {
        if (store.emailTagsCache.base) {
            renderEmailTagsContent(emailTemplate);
        } else {
            showPreloader();
            fetchBaseContent().then(() => renderEmailTagsContent(emailTemplate));
        }
    }

    function fetchBaseContent() {
        var dfd = $.Deferred();

        ui.fetch({
            action: 'mkb_get_email_tags_options'
        }).done(function(response) {
            if (!response || response.status == 1) {
                ui.handleErrors(response);
                dfd.reject();

                return;
            }

            store.emailTagsCache.base = response.html;

            dfd.resolve();
        });

        return dfd;
    }

    function renderEmailTagsContent(emailTemplate) {
        if (!store.emailTagsCache.base) {
            return;
        }

        popup.render({
            title: 'Select Email Tag TK',
            content: store.emailTagsCache.base
        });

        var templateHtml = store.emailTagsCache[emailTemplate];
        var $templateWrap = popup.$el.find('.js-mkb-template-specific-email-tags-wrap');

        if (templateHtml) {
            $templateWrap.html(templateHtml);
        } else {
            ui.fetch({
                action: 'mkb_get_email_template_tags_options',
                template: emailTemplate
            }).done(function(response) {
                if (!response || response.status == 1) {
                    ui.handleErrors(response);

                    return;
                }

                store.emailTagsCache[emailTemplate] = templateHtml = response.html;

                $templateWrap.html(templateHtml);
            });
        }
    }

    /**
     * Renders preloader
     */
    function showPreloader() {
        popup.render({
            title: i18n['loading-options'],
            content: '<div>' +
                '<div class="mkb-loader">' +
                    '<span class="inner1"></span>' +
                    '<span class="inner2"></span>' +
                    '<span class="inner3"></span>' +
                '</div>' +
            '</div>'
        });
    }

    /**
     * Handles shortcode insert click
     */
    function handleEmailTagInsert(e) {
        e.preventDefault();

        editor.execCommand('mceInsertContent', 0, '{{' + e.currentTarget.dataset.insert + '}}');
        popup.close();
    }

    /**
     * Register MCE integration
     */
    tinymce.create('tinymce.plugins.MinervaKBEmail', {
        /**
         * Initializes the plugin, this will be executed after the plugin has been created.
         * This call is done before the editor instance has finished it's initialization so use the onInit event
         * of the editor instance to intercept that event.
         *
         * @param {tinymce.Editor} ed Editor instance that the plugin is initialized in.
         * @param {string} url Absolute URL to where the plugin is located.
         */
        init : function(ed, url) {
            var editorClasses = (ed.settings.editor_class || '').split(' ');

            if (!editorClasses.includes('js-mkb-email-template-editor')) {
                return;
            }

            editor = ed;

            var $editorEl = $(editor.getElement());
            var $settingEl = $editorEl.parents('.fn-control-wrap');
            var emailTemplate = $settingEl.data('email-template')

            ed.addButton('minervakb_email', {
                title : 'Email Template Tags',
                cmd : 'minervakb_email',
                image : url + '/../img/icon-at.svg'
            });

            ed.addCommand('minervakb_email', function() {
                store.selectedText = ed.selection.getContent();

                renderEmailTagsList(emailTemplate);
            });
        },

        /**
         * Creates control instances based on the incoming name. This method is normally not
         * needed since the addButton method of the tinymce.Editor class is a more easy way of adding buttons
         * but you sometimes need to create more complex controls like listboxes, split buttons etc then this
         * method can be used to create those.
         *
         * @param {String} n Name of the control to create.
         * @param {tinymce.ControlManager} cm Control manager to use in order to create new control.
         * @return {tinymce.ui.Control} New control instance or null if no control was created.
         */
        createControl : function(n, cm) {
            return null;
        },

        /**
         * Returns information about the plugin as a name/value array.
         * The current keys are longname, author, authorurl, infourl and version.
         *
         * @return {Object} Name/value array containing information about the plugin.
         */
        getInfo : function() {
            return {
                longname : 'MinervaKB Email Buttons',
                author : 'KonstruktStudio',
                authorurl : 'https://www.minerva-kb.com',
                infourl : 'https://www.minerva-kb.com',
                version : "1.0"
            };
        }
    });

    // Register plugin
    tinymce.PluginManager.add('minervakb_email_plugin', tinymce.plugins.MinervaKBEmail);
})(jQuery);